<?php

class SmartThemeHub
{
    private $api_url;
    private $banners_api_url;

    public function __construct()
    {
        $this->api_url = 'https://ebipour.ir/lamkadeh-updates.json';
        $this->banners_api_url = 'https://ebipour.ir/lamkadeh-banners.json';

        add_action('wp_ajax_get_theme_updates', array($this, 'get_theme_updates_ajax'));
        add_action('wp_ajax_nopriv_get_theme_updates', array($this, 'get_theme_updates_ajax'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }

    public function enqueue_admin_scripts($hook)
    {
        wp_enqueue_style('smart-hub-css', get_stylesheet_directory_uri() . '/admin/assets/styles/smart-hub.css');
        wp_enqueue_script('smart-hub-js', get_stylesheet_directory_uri() . '/admin/assets/scripts/smart-hub.js', array('jquery'), '1.0.0', true);

        wp_localize_script('smart-hub-js', 'smartHub', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('smart_hub_nonce'),
            'i18n' => array(
                'loading' => __('در حال دریافت اطلاعات...', 'lamkadeh'),
                'errorFetch' => __('خطا در دریافت اطلاعات', 'lamkadeh'),
                'serverError' => __('خطا در ارتباط با سرور', 'lamkadeh'),
                'noUpdates' => __('هیچ بروزرسانی‌ای یافت نشد.', 'lamkadeh')
            )
        ));
    }

    public function display_hub_page()
    {
        ob_start();
?>
        <div class="wrap smart-theme-hub">
            <div class="hub-header">
                <h1><?php echo __('هاب هوشمند قالب', 'lamkadeh'); ?></h1>
                <p><?php echo __('مرکز بروزرسانی‌ها و اطلاع‌رسانی‌های قالب', 'lamkadeh'); ?></p>
            </div>

            <div class="hub-controls">
                <div class="api-url-setting">
                    <button type="button" class="button button-primary" id="refresh_updates"><?php echo __('بروزرسانی لیست', 'lamkadeh'); ?></button>
                </div>

                <div class="filter-controls">
                    <button type="button" class="button filter-btn active" data-filter="all"><?php echo __('همه', 'lamkadeh'); ?></button>
                    <button type="button" class="button filter-btn" data-filter="released"><?php echo __('بروزرسانی شده', 'lamkadeh'); ?></button>
                    <button type="button" class="button filter-btn" data-filter="bugfix"><?php echo __('رفع باگ', 'lamkadeh'); ?></button>
                    <button type="button" class="button filter-btn" data-filter="planned"><?php echo __('برنامه‌ریزی شده', 'lamkadeh'); ?></button>
                </div>
            </div>

            <div class="updates-container" id="updates_container">
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p><?php echo __('در حال دریافت اطلاعات...', 'lamkadeh'); ?></p>
                </div>
            </div>

            <div class="banners-container" id="banners_container">
                <!-- بنرها اینجا نمایش داده می‌شوند -->
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function get_theme_updates_ajax()
    {
        check_ajax_referer('smart_hub_nonce', 'nonce');

        $api_url = isset($_POST['api_url']) ? esc_url_raw($_POST['api_url']) : $this->api_url;
        $banners_url = isset($_POST['banners_url']) ? esc_url_raw($_POST['banners_url']) : $this->banners_api_url;

        $updates = $this->fetch_api_data($api_url);
        $banners = $this->fetch_api_data($banners_url);

        if (is_wp_error($updates)) {
            wp_send_json_error(__('خطا در دریافت اطلاعات بروزرسانی‌ها', 'lamkadeh'));
        }

        if (is_wp_error($banners)) {
            wp_send_json_error(__('خطا در دریافت بنرها', 'lamkadeh'));
        }

        ob_start();
        $this->render_updates($updates);
        $updates_html = ob_get_clean();

        ob_start();
        $this->render_banners($banners);
        $banners_html = ob_get_clean();

        wp_send_json_success(array(
            'updates' => $updates_html,
            'banners' => $banners_html
        ));
    }

    private function fetch_api_data($url)
    {
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'sslverify' => false
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('invalid_json', __('داده‌های دریافتی معتبر نیستند', 'lamkadeh'));
        }

        return $data;
    }

    private function render_updates($updates)
    {
        if (empty($updates)) {
            echo '<div class="no-updates">' . __('هیچ بروزرسانی‌ای یافت نشد.', 'lamkadeh') . '</div>';
            return;
        }

        foreach ($updates as $update) {
            $status_class = $this->get_status_class($update['status']);
            $status_text = $this->get_status_text($update['status']);
        ?>
            <div class="update-card <?php echo esc_attr($status_class); ?>" data-status="<?php echo esc_attr($update['status']); ?>">
                <div class="update-header">
                    <div class="version-badge">
                        <span class="version"><?php echo esc_html($update['version']); ?></span>
                        <span class="status-badge <?php echo esc_attr($status_class); ?>"><?php echo esc_html($status_text); ?></span>
                    </div>
                    <div class="update-date"><?php echo esc_html($update['date']); ?></div>
                </div>

                <div class="update-content">
                    <h3 class="update-title"><?php echo esc_html($update['title']); ?></h3>
                    <p class="update-description"><?php echo esc_html($update['description']); ?></p>

                    <?php if (!empty($update['features'])): ?>
                        <div class="features-list">
                            <h4><?php echo __('ویژگی‌های این بروزرسانی:', 'lamkadeh'); ?></h4>
                            <ul>
                                <?php foreach ($update['features'] as $feature): ?>
                                    <li><?php echo esc_html($feature); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <div class="update-meta">
                        <div class="category"><?php echo __('دسته:', 'lamkadeh'); ?> <?php echo esc_html($update['category']); ?></div>
                        <div class="feedback">
                            <span class="likes">👍 <?php echo intval($update['likes']); ?></span>
                            <span class="dislikes">👎 <?php echo intval($update['dislikes']); ?></span>
                        </div>
                    </div>

                    <?php if (!empty($update['changelog'])): ?>
                        <div class="update-actions">
                            <a href="<?php echo esc_url($update['changelog']); ?>" target="_blank" class="button changelog-btn"><?php echo __('مشاهده تغییرات کامل', 'lamkadeh'); ?></a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php
        }
    }

    private function render_banners($banners)
    {
        if (empty($banners)) {
            return;
        }

        echo '<div class="banners-grid">';
        foreach ($banners as $banner) {
            $alt_text = $banner['alt'] ?? __('بنر تبلیغاتی', 'lamkadeh');
        ?>
            <div class="banner-item">
                <a href="<?php echo esc_url($banner['link']); ?>" target="_blank" class="banner-link">
                    <img src="<?php echo esc_url($banner['image']); ?>" alt="<?php echo esc_attr($alt_text); ?>" class="banner-image">
                    <?php if (!empty($banner['title'])): ?>
                        <div class="banner-title"><?php echo esc_html($banner['title']); ?></div>
                    <?php endif; ?>
                </a>
            </div>
<?php
        }
        echo '</div>';
    }

    private function get_status_class($status)
    {
        $status_map = array(
            'released' => 'released',
            'bugfix' => 'bugfix',
            'planned' => 'planned'
        );

        return $status_map[$status] ?? 'released';
    }

    private function get_status_text($status)
    {
        $status_texts = array(
            'released' => __('بروزرسانی شده', 'lamkadeh'),
            'bugfix' => __('رفع باگ', 'lamkadeh'),
            'planned' => __('برنامه‌ریزی شده', 'lamkadeh')
        );

        return $status_texts[$status] ?? __('بروزرسانی شده', 'lamkadeh');
    }
}

new SmartThemeHub();

function smart_hub_shortcode($atts)
{
    $hub = new SmartThemeHub();
    return $hub->display_hub_page();
}
add_shortcode("hub_page", "smart_hub_shortcode");

function smart_hub()
{
    echo do_shortcode("[hub_page]");
}
