<?php

namespace Elementor;

if (!defined('ABSPATH'))
    exit;

class CategoryProducts extends Widget_Base
{
    public function get_name()
    {
        return 'CategoryProducts';
    }

    public function get_title()
    {
        return __('دسته بندی بر اساس محصولات لمکده', 'lamkadeh');
    }

    public function get_icon()
    {
        return 'eicon-product-categories';
    }

    public function get_categories()
    {
        return ['lamkadeh'];
    }

    public function register_controls()
    {
        $this->start_controls_section(
            'category_widget_settings',
            [
                'label' => __('تنظیمات دسته‌بندی ها بر اساس محصولات', 'lamkadeh'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'categories_title',
            [
                'label' => __('عنوان دسته بندی ها', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('دسته بندی محصولات', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'categories_title_en',
            [
                'label' => __('عنوان انگلیسی دسته بندی ها', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Categories', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'categories_description',
            [
                'label' => __('توضیحات دسته بندی', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::WYSIWYG,
                'default' => __('توضیحات کوتاه درباره دسته بندی ها', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'categories_link_text',
            [
                'label' => __('متن لینک مشاهده دسته بندی ها', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('مشاهده همه', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'categories_link_url',
            [
                'label' => __('لینک مشاهده همه دسته بندی ها', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::URL,
                'show_external' => true,
                'default' => ['url' => '', 'is_external' => true, 'nofollow' => true],
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'selected_category',
            [
                'label' => __('انتخاب دسته‌بندی', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::SELECT2,
                'options' => $this->get_product_categories(),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'categories_list',
            [
                'label' => __('دسته‌بندی‌های انتخابی', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [],
                'title_field' => '{{{ selected_category }}}',
            ]
        );

        $this->end_controls_section();
    }

    // تابع برای محدود کردن تعداد آیتم‌های ریپیتر
    public function __construct($data = [], $args = null)
    {
        parent::__construct($data, $args);

        // محدود کردن ریپیتر به 3 آیتم
        add_action('elementor/editor/after_enqueue_scripts', function () {
            ?>
            <script>
                jQuery(document).ready(function ($) {
                    // محدودیت برای ویجت CategoryProducts
                    var limitRepeater = function () {
                        var widgetID = '<?php echo $this->get_id(); ?>';

                        elementor.hooks.addFilter('elements/add-repeater-item', function (shouldAdd, model, view) {
                            var widgetType = model.get('widgetType');
                            var currentItems = model.get('settings').categories_list ? model.get('settings').categories_list.length : 0;

                            // اگر ویجت ما باشد و بیشتر از 3 آیتم داشته باشد
                            if (widgetType === 'CategoryProducts' && model.get('settings').categories_list && currentItems >= 3) {
                                alert('شما نمی‌توانید بیشتر از 3 دسته‌بندی اضافه کنید.');
                                return false;
                            }

                            return shouldAdd;
                        });
                    };

                    // اجرا پس از لود کامل المنتور
                    if (typeof elementor !== 'undefined') {
                        limitRepeater();
                    } else {
                        $(window).on('elementor:init', limitRepeater);
                    }
                });
            </script>
            <?php
        });
    }

    // تابع برای دریافت دسته‌بندی‌های محصولات ووکامرس
    private function get_product_categories()
    {
        $categories = [];

        if (!taxonomy_exists('product_cat')) {
            return $categories;
        }

        $terms = get_terms([
            'taxonomy' => 'product_cat',
            'hide_empty' => true,
        ]);

        if (!empty($terms) && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[$term->slug] = $term->name;
            }
        }

        return $categories;
    }

    // تابع برای دریافت محصولات جدید یک دسته‌بندی
    private function get_category_products($category_slug, $limit = 4)
    {
        if (empty($category_slug) || !function_exists('wc_get_products')) {
            return [];
        }

        $args = [
            'status' => 'publish',
            'limit' => $limit,
            'category' => [$category_slug],
            'orderby' => 'date',
            'order' => 'DESC',
            'no_found_rows'          => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ];

        $products = wc_get_products($args);
        $products_data = [];

        foreach ($products as $index => $product) {
            $products_data[] = [
                'id' => $product->get_id(),
                'name' => $product->get_name(),
                'permalink' => $product->get_permalink(),
                'image_url' => get_the_post_thumbnail_url($product->get_id(), 'medium'),
                'is_new' => $index === 0, // فقط اولین محصول جدید است
            ];
        }

        return $products_data;
    }

    // تابع برای دریافت تعداد محصولات یک دسته‌بندی
    private function get_category_product_count($category_slug)
    {
        if (empty($category_slug)) {
            return 0;
        }

        $term = get_term_by('slug', $category_slug, 'product_cat');
        if ($term && !is_wp_error($term)) {
            return $term->count;
        }

        return 0;
    }

    // تابع برای دریافت اطلاعات دسته‌بندی
    private function get_category_info($category_slug)
    {
        if (empty($category_slug)) {
            return null;
        }

        $term = get_term_by('slug', $category_slug, 'product_cat');
        if ($term && !is_wp_error($term)) {
            return [
                'name' => $term->name,
                'description' => $term->description,
                'count' => $term->count,
                'url' => get_term_link($term),
                'thumbnail_id' => get_term_meta($term->term_id, 'thumbnail_id', true),
            ];
        }

        return null;
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $categories_list = $settings['categories_list'];

        if (is_array($categories_list) && count($categories_list) > 3) {
            $categories_list = array_slice($categories_list, 0, 3);
        }
        ?>
        <section class="container">
            <section class="category-products">
                <div class="header-categories-main">
                    <div class="title-header-categories-main">
                        <span><?php echo esc_html($settings['categories_title_en']) ?></span>
                        <div class="text-header-categories-main">
                            <span><?php echo esc_html($settings['categories_title']) ?></span>
                            <svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <g clip-path="url(#clip0_1_3203)">
                                    <circle cx="32" cy="32" r="32" transform="matrix(-1 0 0 1 64 0)" fill="#FFD701" />
                                    <path fill-rule="evenodd" clip-rule="evenodd"
                                        d="M42.0002 12.1458C41.4458 12.1458 40.9354 12.2835 40.3765 12.5146C39.8369 12.7378 39.2101 13.0667 38.4297 13.4763L36.5344 14.4709C35.5765 14.9736 34.81 15.3758 34.2181 15.7711C33.6049 16.1806 33.1326 16.6118 32.7887 17.1958C32.4459 17.7779 32.2917 18.4083 32.2178 19.1589C32.146 19.8875 32.146 20.7836 32.146 21.9106V22.0892C32.146 23.2162 32.146 24.1123 32.2178 24.8409C32.2917 25.5916 32.4459 26.2219 32.7887 26.804C33.1326 27.3881 33.6049 27.8192 34.2181 28.2287C34.81 28.624 35.5765 29.0263 36.5344 29.5289L38.4296 30.5235C39.2101 30.9331 39.8369 31.262 40.3765 31.4852C40.9354 31.7163 41.4458 31.8541 42.0002 31.8541C42.5545 31.8541 43.0649 31.7163 43.6238 31.4852C44.1634 31.262 44.7902 30.9331 45.5706 30.5235L47.4659 29.529C48.4238 29.0263 49.1903 28.624 49.7822 28.2287C50.3954 27.8192 50.8677 27.3881 51.2116 26.804C51.5544 26.2219 51.7086 25.5916 51.7826 24.8409C51.8543 24.1123 51.8543 23.2162 51.8543 22.0892V21.9106C51.8543 20.7836 51.8543 19.8875 51.7826 19.1589C51.7086 18.4083 51.5544 17.7779 51.2116 17.1958C50.8677 16.6118 50.3954 16.1806 49.7822 15.7711C49.1903 15.3758 48.4238 14.9735 47.4659 14.4709L45.5707 13.4763C44.7902 13.0667 44.1634 12.7378 43.6238 12.5146C43.0649 12.2835 42.5545 12.1458 42.0002 12.1458ZM39.0401 14.7088C39.8556 14.2808 40.4272 13.9816 40.902 13.7853C41.3642 13.5941 41.6889 13.5208 42.0002 13.5208C42.3114 13.5208 42.6361 13.5941 43.0983 13.7853C43.5732 13.9816 44.1447 14.2808 44.9603 14.7088L46.7936 15.6709C47.7925 16.1951 48.4943 16.5644 49.0186 16.9145C49.2773 17.0873 49.4806 17.2477 49.6447 17.409L46.5915 18.9356L38.8 14.8348L39.0401 14.7088ZM37.3661 15.5873L37.2067 15.6709C36.2078 16.1951 35.5061 16.5644 34.9817 16.9145C34.723 17.0873 34.5198 17.2477 34.3557 17.409L42.0002 21.2313L45.0776 19.6926L37.555 15.7333C37.4819 15.6948 37.4186 15.6451 37.3661 15.5873ZM33.693 18.615C33.6467 18.8111 33.6118 19.0337 33.5861 19.2937C33.5218 19.9471 33.521 20.7756 33.521 21.9463V22.0535C33.521 23.2242 33.5218 24.0527 33.5861 24.7061C33.6491 25.3452 33.7687 25.7585 33.9735 26.1063C34.1772 26.4522 34.4718 26.7448 34.9817 27.0853C35.5061 27.4355 36.2078 27.8047 37.2067 28.3289L39.0401 29.291C39.8556 29.719 40.4272 30.0182 40.902 30.2146C41.052 30.2766 41.1874 30.3262 41.3127 30.3649V22.4248L33.693 18.615ZM42.6877 30.3649C42.8129 30.3262 42.9484 30.2766 43.0983 30.2146C43.5732 30.0182 44.1447 29.719 44.9603 29.291L46.7936 28.3289C47.7925 27.8047 48.4943 27.4355 49.0186 27.0853C49.5285 26.7448 49.8231 26.4522 50.0268 26.1063C50.2316 25.7585 50.3512 25.3452 50.4142 24.7061C50.4786 24.0527 50.4793 23.2242 50.4793 22.0535V21.9463C50.4793 20.7756 50.4786 19.9471 50.4142 19.2937C50.3886 19.0337 50.3536 18.8111 50.3073 18.615L47.271 20.1331V22.9166C47.271 23.2963 46.9632 23.6041 46.5835 23.6041C46.2038 23.6041 45.896 23.2963 45.896 22.9166V20.8206L42.6877 22.4248V30.3649Z"
                                        fill="#1D2977" />
                                    <circle cx="22" cy="60" r="17" fill="#F7F8FD" fill-opacity="0.64" />
                                </g>
                                <defs>
                                    <clipPath id="clip0_1_3203">
                                        <rect width="64" height="64" rx="32" fill="white" />
                                    </clipPath>
                                </defs>
                            </svg>
                        </div>
                        <div class="shape-header-categories-main"></div>
                    </div>
                    <div class="line-header-categories-main"></div>
                    <p class="description-header-categories-main">
                        <?php echo wp_kses($settings['categories_description'], [
                            'a' => ['href' => [], 'title' => [], 'target' => []],
                            'br' => [],
                            'strong' => [],
                            'em' => [],
                            'b' => [],
                            'i' => [],
                            'span' => ['class' => []],
                        ]); ?>
                    </p>
                    <div class="line-header-categories-main-two"></div>
                    <a href="<?php echo esc_url($settings['categories_link_url']['url']) ?>"
                        class="show-more-header-categories-main">
                        <div>
                            <span><?php echo esc_html($settings['categories_link_text']) ?></span>
                        </div>
                        <div>
                            <svg width="40" height="40" viewBox="0 0 40 40" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path
                                    d="M23.6216 13.6066C23.4648 13.7633 23.2504 13.8623 23.0029 13.8623L13.8623 13.8623L13.8623 23.0029C13.8623 23.4813 13.4664 23.8773 12.9879 23.8773C12.5094 23.8773 12.1134 23.4813 12.1134 23.0029L12.1134 12.9879C12.1134 12.5094 12.5094 12.1134 12.9879 12.1134L23.0029 12.1134C23.4813 12.1134 23.8773 12.5094 23.8773 12.9879C23.8856 13.2271 23.7783 13.4498 23.6216 13.6066Z"
                                    fill="white" />
                                <path
                                    d="M27.6308 27.6308C27.2926 27.969 26.7316 27.969 26.3934 27.6308L12.5093 13.7468C12.1711 13.4085 12.1711 12.8476 12.5093 12.5093C12.8476 12.1711 13.4085 12.1711 13.7468 12.5093L27.6308 26.3934C27.9691 26.7316 27.9691 27.2926 27.6308 27.6308Z"
                                    fill="white" />
                            </svg>
                        </div>
                    </a>
                </div>

                <div class="category-products-body">
                    <?php if (!empty($categories_list)):
                        foreach ($categories_list as $index => $category_item):
                            $category_slug = $category_item['selected_category'];
                            $category_info = $this->get_category_info($category_slug);
                            $category_products = $this->get_category_products($category_slug, 4);
                            $term = get_term_by('slug', $category_slug, 'product_cat');
                            $category_link = get_term_link($term);

                            if (!$category_info || empty($category_products)) {
                                continue;
                            }

                            $style = ($index >= 3) ? 'style="display:none;"' : '';
                            ?>
                            <div class="category-products-body-items">
                                <div class="category-products-body-items-header">
                                    <div class="category-products-body-items-header-right">
                                        <div class="category-products-body-items-header-right-shape"></div>
                                        <a
                                            class="category-products-body-items-header-right-title" href="<?php echo esc_url($category_link); ?>"><?php echo esc_html($category_info['name']); ?></a>
                                        <div class="category-products-body-items-header-right-content">
                                            <span class="category-products-body-items-header-right-content-text">
                                                <span><strong><?php echo esc_html($category_info['count']); ?></strong> محصول</span>
                                            </span>
                                            <div class="category-products-body-items-header-right-content-line"></div>
                                            <div class="category-products-body-items-header-right-content-arrow">
                                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none"
                                                    xmlns="http://www.w3.org/2000/svg">
                                                    <path fill-rule="evenodd" clip-rule="evenodd"
                                                        d="M10.1023 4.10225C9.88258 4.32192 9.88258 4.67808 10.1023 4.89775L13.642 8.4375L3 8.4375C2.68934 8.4375 2.4375 8.68934 2.4375 9C2.4375 9.31066 2.68934 9.5625 3 9.5625L13.642 9.5625L10.1023 13.1023C9.88258 13.3219 9.88258 13.6781 10.1023 13.8977C10.3219 14.1174 10.6781 14.1174 10.8977 13.8977L15.3977 9.39775C15.6174 9.17808 15.6174 8.82192 15.3977 8.60225L10.8977 4.10225C10.6781 3.88258 10.3219 3.88258 10.1023 4.10225Z"
                                                        fill="#020308" fill-opacity="0.64" />
                                                </svg>
                                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none"
                                                    xmlns="http://www.w3.org/2000/svg">
                                                    <path fill-rule="evenodd" clip-rule="evenodd"
                                                        d="M7.89775 4.10225C8.11742 4.32192 8.11742 4.67808 7.89775 4.89775L4.358 8.4375L15 8.4375C15.3107 8.4375 15.5625 8.68934 15.5625 9C15.5625 9.31066 15.3107 9.5625 15 9.5625L4.358 9.5625L7.89775 13.1023C8.11742 13.3219 8.11742 13.6781 7.89775 13.8977C7.67808 14.1174 7.32192 14.1174 7.10225 13.8977L2.60225 9.39775C2.38258 9.17808 2.38258 8.82192 2.60225 8.60225L7.10225 4.10225C7.32192 3.88258 7.67808 3.88258 7.89775 4.10225Z"
                                                        fill="#020308" fill-opacity="0.64" />
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="category-products-body-items-header-left">
                                        <?php
                                        // نمایش 3 تصویر اول از محصولات
                                        $display_images = array_slice($category_products, 0, 3);
                                        foreach ($display_images as $product):
                                            ?>
                                            <img src="<?php echo esc_url($product['image_url'] ?: 'https://via.placeholder.com/300x300'); ?>"
                                                alt="<?php echo esc_attr($product['name']); ?>">
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                <div class="category-products-body-items-bottom">
                                    <div class="category-products-body-items-bottom-top">
                                        <?php
                                        // محصولات 1 و 2
                                        $products_1_2 = array_slice($category_products, 0, 2);
                                        foreach ($products_1_2 as $key => $product):
                                            ?>
                                            <a href="<?php echo esc_url($product['permalink']); ?>"
                                                class="category-products-body-items-bottom-top-item"
                                                title="<?php echo esc_attr($product['name']); ?>">
                                                <span><?php echo sprintf('%02d', $key + 1); ?></span>
                                                <img src="<?php echo esc_url($product['image_url'] ?: 'https://via.placeholder.com/300x300'); ?>"
                                                    alt="<?php echo esc_attr($product['name']); ?>">
                                                <?php if ($product['is_new']): ?>
                                                    <div>
                                                        <span>جدید</span>
                                                    </div>
                                                <?php endif; ?>
                                                <div class="category-products-body-items-bottom-bottom-svg">
                                                    <svg width="29" height="29" viewBox="0 0 29 29" fill="none"
                                                        xmlns="http://www.w3.org/2000/svg">
                                                        <path fill-rule="evenodd" clip-rule="evenodd"
                                                            d="M16.3971 8.97965C16.4404 9.3221 16.1979 9.63482 15.8554 9.67813L10.3372 10.376L19.6814 17.622C19.9542 17.8335 20.0038 18.2261 19.7923 18.4989C19.5808 18.7717 19.1882 18.8213 18.9154 18.6098L9.57122 11.3638L10.2691 16.8821C10.3124 17.2245 10.0699 17.5372 9.72748 17.5805C9.38503 17.6238 9.07231 17.3813 9.029 17.0389L8.14178 10.0237C8.09847 9.68126 8.34097 9.36854 8.68342 9.32523L15.6986 8.43801C16.0411 8.3947 16.3538 8.6372 16.3971 8.97965Z"
                                                            fill="#1D2977" />
                                                    </svg>
                                                </div>
                                            </a>
                                        <?php endforeach; ?>
                                    </div>
                                    <div class="category-products-body-items-bottom-bottom">
                                        <?php
                                        // محصولات 3 و 4
                                        $products_3_4 = array_slice($category_products, 2, 2);
                                        foreach ($products_3_4 as $key => $product):
                                            ?>
                                            <a href="<?php echo esc_url($product['permalink']); ?>"
                                                class="category-products-body-items-bottom-top-item"
                                                title="<?php echo esc_attr($product['name']); ?>">
                                                <span><?php echo sprintf('%02d', $key + 3); ?></span>
                                                <img src="<?php echo esc_url($product['image_url'] ?: 'https://via.placeholder.com/300x300'); ?>"
                                                    alt="<?php echo esc_attr($product['name']); ?>">
                                                <div class="category-products-body-items-bottom-bottom-svg">
                                                    <svg width="29" height="29" viewBox="0 0 29 29" fill="none"
                                                        xmlns="http://www.w3.org/2000/svg">
                                                        <path fill-rule="evenodd" clip-rule="evenodd"
                                                            d="M16.3971 8.97965C16.4404 9.3221 16.1979 9.63482 15.8554 9.67813L10.3372 10.376L19.6814 17.622C19.9542 17.8335 20.0038 18.2261 19.7923 18.4989C19.5808 18.7717 19.1882 18.8213 18.9154 18.6098L9.57122 11.3638L10.2691 16.8821C10.3124 17.2245 10.0699 17.5372 9.72748 17.5805C9.38503 17.6238 9.07231 17.3813 9.029 17.0389L8.14178 10.0237C8.09847 9.68126 8.34097 9.36854 8.68342 9.32523L15.6986 8.43801C16.0411 8.3947 16.3538 8.6372 16.3971 8.97965Z"
                                                            fill="#1D2977" />
                                                    </svg>
                                                </div>
                                            </a>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            <div class="underline-lamkadeh" <?= $style ?>>
                                <div class="underline-lamkadeh-line"></div>
                                <div class="underline-lamkadeh-circle"></div>
                            </div>
                            <div class="category-products-body-line">
                                <div class="category-products-body-line-circle"></div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p><?php _e('لطفا دسته‌بندی‌هایی را انتخاب کنید.', 'lamkadeh'); ?></p>
                    <?php endif; ?>
                </div>

                <a href="<?php echo esc_url($settings['categories_link_url']['url']) ?>"
                    class="bottom-show-more-header-categories-main">
                    <div>
                        <span><?php echo esc_html($settings['categories_link_text']) ?></span>
                    </div>
                    <div>
                        <svg width="40" height="40" viewBox="0 0 40 40" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path
                                d="M23.6216 13.6066C23.4648 13.7633 23.2504 13.8623 23.0029 13.8623L13.8623 13.8623L13.8623 23.0029C13.8623 23.4813 13.4664 23.8773 12.9879 23.8773C12.5094 23.8773 12.1134 23.4813 12.1134 23.0029L12.1134 12.9879C12.1134 12.5094 12.5094 12.1134 12.9879 12.1134L23.0029 12.1134C23.4813 12.1134 23.8773 12.5094 23.8773 12.9879C23.8856 13.2271 23.7783 13.4498 23.6216 13.6066Z"
                                fill="white" />
                            <path
                                d="M27.6308 27.6308C27.2926 27.969 26.7316 27.969 26.3934 27.6308L12.5093 13.7468C12.1711 13.4085 12.1711 12.8476 12.5093 12.5093C12.8476 12.1711 13.4085 12.1711 13.7468 12.5093L27.6308 26.3934C27.9691 26.7316 27.9691 27.2926 27.6308 27.6308Z"
                                fill="white" />
                        </svg>
                    </div>
                </a>
            </section>
        </section>
        <?php
    }
}
