<?php

remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_rating', 10);

add_action('woocommerce_single_product_summary', 'woocommerce_template_single_rating', 4);

remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_meta', 10);

add_action('woocommerce_single_product_summary', 'lamkadeh_show_product_categories', 4);

add_filter('woocommerce_product_tabs', 'reviews_tab_title', 20);
function reviews_tab_title($tabs)
{
    if (isset($tabs['reviews'])) {
        $tabs['reviews']['title'] = __('نظرات شما', "lamkadeh");
    }
    return $tabs;
}

function lamkadeh_show_product_categories()
{
    global $product;

    $categories = get_the_terms($product->get_id(), 'product_cat');

    if ($categories && !is_wp_error($categories)) {
        $main_category = null;
        foreach ($categories as $category) {
            if ($category->parent == 0) {
                $main_category = $category;
                break;
            }
        }

        if (!$main_category) {
            $main_category = reset($categories);
        }

        if ($main_category) {
            $thumbnail_id = get_term_meta($main_category->term_id, 'thumbnail_id', true);
            $image_url = $thumbnail_id
                ? wp_get_attachment_image_url($thumbnail_id, 'thumbnail')
                : wc_placeholder_img_src();

            echo '<div class="lamkadeh-main-category" style="display:flex;align-items:center;gap:8px;margin-bottom:10px;margin-top:25px">';
            echo '<a href="' . esc_url(get_term_link($main_category)) . '" class="lamkadeh-main-category-link" style="display:flex;align-items:center;gap:5px;text-decoration:none;color:inherit;">';
            echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($main_category->name) . '" style="width:20px;height:20px;object-fit:cover;border-radius:4px;">';
            echo '<span>' . esc_html($main_category->name) . '</span>';
            echo '</a>';
            echo '</div>';
        }
    }
}

add_filter('woocommerce_kses_notice_allowed_tags', function ($tags) {
    $tags['script'] = [
        'src' => true,
        'type' => true,
        'id' => true,
    ];
    return $tags;
});

remove_action('woocommerce_before_single_product_summary', 'woocommerce_show_product_sale_flash', 10);

add_action('woocommerce_single_product_summary', 'woocommerce_show_product_sale_flash', 3);

remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_meta', 40);

remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_excerpt', 20);

add_action('woocommerce_after_add_to_cart_button', 'show_short_description_below_add_to_cart');

function show_short_description_below_add_to_cart()
{
    global $product;

    if ($product) {
        $short_description = apply_filters('woocommerce_short_description', $product->get_short_description());

        if ($short_description) {
            echo '<div class="custom-short-description" style="margin:15px;">' . wp_kses_post($short_description) . '</div>';
        }
    }
}

add_action('add_meta_boxes', 'lamkadeh_add_special_stock_box');
function lamkadeh_add_special_stock_box()
{
    add_meta_box(
        'lamkadeh_special_stock',
        __("باکس ویژه محصول", "lamkadeh"),
        'lamkadeh_special_callback',
        'product',
        'normal',
        'high'
    );
}

function lamkadeh_special_callback($post)
{
    $short_code_banner_product = get_post_meta($post->ID, '_short_code_banner_product', true);
    ?>
    <br>
    <label for="short_code_banner_product">شورت کد بنر تبلیغاتی محصول: </label>
    <input type="text" name="short_code_banner_product" style="width:100%;direction: ltr;" id="short_code_banner_product"
        value="<?php echo esc_attr($short_code_banner_product) ?>" />
    <?php
}

add_action('save_post', 'lamkadeh_save_special_stock_data');
function lamkadeh_save_special_stock_data($post_id)
{
    if (isset($_POST['lamkadeh_special_stock_field'])) {
        update_post_meta($post_id, '_short_code_banner_product', sanitize_text_field($_POST['short_code_banner_product']));
    }
}

// ==================== HELPER FUNCTIONS ====================

/**
 * Get product ID from order item (safe with fallback)
 */
if (!function_exists('lamkadeh_get_item_product_id')) {
    function lamkadeh_get_item_product_id($item)
    {
        // WC_Order_Item_Product object
        if (is_object($item)) {
            if (method_exists($item, 'get_product_id')) {
                return (int) $item->get_product_id();
            }
            if (method_exists($item, 'get_variation_id')) {
                return (int) $item->get_variation_id();
            }
            if (method_exists($item, 'get_product')) {
                $prod = $item->get_product();
                return $prod ? (int) $prod->get_id() : 0;
            }
            // Fallback to properties
            if (isset($item->product_id))
                return (int) $item->product_id;
            if (isset($item->variation_id))
                return (int) $item->variation_id;
        }

        // Array item
        if (is_array($item)) {
            if (!empty($item['product_id']))
                return (int) $item['product_id'];
            if (!empty($item['variation_id']))
                return (int) $item['variation_id'];
        }

        return 0;
    }
}

/**
 * Get unique buyers for a product with transient cache
 */
if (!function_exists('lamkadeh_get_buyers_for_product')) {
    function lamkadeh_get_buyers_for_product($product_id, $cache_seconds = 300)
    {
        $product_id = (int) $product_id;
        if ($product_id <= 0)
            return array();

        if (!function_exists('wc_get_orders'))
            return array();

        $transient_key = 'lamkadeh_buyers_' . $product_id;
        $cached = get_transient($transient_key);
        if ($cached !== false)
            return $cached;

        $buyers = array(); // 'u{user_id}' or 'e{md5(email)}' => array(type=>..., id/email=>...)

        $orders = wc_get_orders(array(
            'limit' => -1,
            'status' => array('completed', 'processing', 'on-hold'),
            'return' => 'ids',
        ));

        if (!empty($orders) && is_array($orders)) {
            foreach ($orders as $order_id) {
                $order = wc_get_order($order_id);
                if (!$order)
                    continue;

                $items = $order->get_items();
                if (empty($items))
                    continue;

                foreach ($items as $item) {
                    $item_product_id = lamkadeh_get_item_product_id($item);
                    if ($item_product_id === $product_id) {
                        $user_id = (int) $order->get_user_id();
                        if ($user_id > 0) {
                            $buyers['u' . $user_id] = array('type' => 'user', 'id' => $user_id);
                        } else {
                            $email = $order->get_billing_email();
                            if ($email) {
                                $key = 'e' . md5(strtolower(trim($email)));
                                $buyers[$key] = array('type' => 'email', 'email' => $email);
                            }
                        }
                    }
                }
            }
        }

        set_transient($transient_key, $buyers, $cache_seconds);
        return $buyers;
    }
}

/**
 * Check if special stock should be shown for a product
 */
if (!function_exists('lamkadeh_should_show_special_stock')) {
    function lamkadeh_should_show_special_stock($product)
    {
        if (empty($product) || !is_a($product, 'WC_Product')) {
            return false;
        }

        // Simple product
        if ($product->is_type('simple')) {
            $special_stock = (int) get_post_meta($product->get_id(), '_lamkadeh_special_stock', true);

            // Don't show if special stock is zero/negative or sold out
            if ($special_stock <= 0)
                return false;

            $total_sold = (int) $product->get_total_sales();
            if ($total_sold >= $special_stock)
                return false;

            return true;
        }

        // Variable product - check if any variation has available special stock
        if ($product->is_type('variable')) {
            $variations = $product->get_available_variations();

            foreach ($variations as $variation) {
                $variation_id = $variation['variation_id'];
                $special_stock = (int) get_post_meta($variation_id, '_lamkadeh_special_stock', true);

                if ($special_stock > 0) {
                    // FIX: Use get_post_meta for variation sales
                    $total_sold = (int) get_post_meta($variation_id, 'total_sales', true);
                    if ($total_sold < $special_stock) {
                        return true;
                    }
                }
            }
        }

        return false;
    }
}

/**
 * Get max special stock values for variable products
 */
if (!function_exists('lamkadeh_get_variable_product_max_values')) {
    function lamkadeh_get_variable_product_max_values($product)
    {
        $max_values = array(
            'special_stock' => 0,
            'total_sold' => 0,
            'remaining' => 0,
            'percentage_sold' => 0
        );

        if (!$product->is_type('variable'))
            return $max_values;

        $variations = $product->get_available_variations();
        $max_special_stock = 0;
        $max_total_sold = 0;

        foreach ($variations as $variation) {
            $variation_id = $variation['variation_id'];
            $special_stock = (int) get_post_meta($variation_id, '_lamkadeh_special_stock', true);
            // FIX: Use get_post_meta for variation sales
            $total_sold = (int) get_post_meta($variation_id, 'total_sales', true);

            // Only consider variations with available special stock
            if ($special_stock > 0 && $total_sold < $special_stock) {
                if ($special_stock > $max_special_stock) {
                    $max_special_stock = $special_stock;
                    $max_total_sold = $total_sold;
                }
            }
        }

        if ($max_special_stock > 0) {
            $remaining = max(0, $max_special_stock - $max_total_sold);
            $percentage_sold = ($max_special_stock > 0) ? ($max_total_sold / $max_special_stock) * 100 : 0;
            $percentage_sold = min(100, max(0, round($percentage_sold, 2)));

            $max_values = array(
                'special_stock' => $max_special_stock,
                'total_sold' => $max_total_sold,
                'remaining' => $remaining,
                'percentage_sold' => $percentage_sold
            );
        }

        return $max_values;
    }
}

// ==================== ADMIN FIELDS ====================

/**
 * Add special stock field to simple products
 */
add_action('woocommerce_product_options_inventory_product_data', 'lamkadeh_add_special_stock_field');
function lamkadeh_add_special_stock_field()
{
    global $product_object;

    if ($product_object && $product_object->is_type('simple')) {
        woocommerce_wp_text_input(array(
            'id' => '_lamkadeh_special_stock',
            'label' => __('Promotion stock quantity', 'lamkadeh'),
            'description' => __('Total special stock quantity for this product', 'lamkadeh'),
            'desc_tip' => true,
            'type' => 'number',
            'custom_attributes' => array(
                'min' => '0',
                'step' => '1'
            )
        ));
    }
}

/**
 * Add special stock field to variations
 */
add_action('woocommerce_product_after_variable_attributes', 'lamkadeh_add_special_stock_to_variations', 10, 3);
function lamkadeh_add_special_stock_to_variations($loop, $variation_data, $variation)
{
    $special_stock = get_post_meta($variation->ID, '_lamkadeh_special_stock', true);
    ?>
    <div class="form-row form-row-full">
        <h4><?php echo esc_html__('Special stock', 'lamkadeh'); ?></h4>
        <?php
        woocommerce_wp_text_input(array(
            'id' => '_lamkadeh_special_stock[' . $loop . ']',
            'name' => '_lamkadeh_special_stock[' . $loop . ']',
            'label' => __('Special stock quantity', 'lamkadeh'),
            'value' => $special_stock,
            'type' => 'number',
            'custom_attributes' => array(
                'min' => '0',
                'step' => '1'
            ),
            'wrapper_class' => 'form-row form-row-full',
        ));
        ?>
    </div>
    <?php
}

/**
 * Save special stock for simple products
 */
add_action('woocommerce_admin_process_product_object', 'lamkadeh_save_special_stock_field');
function lamkadeh_save_special_stock_field($product)
{
    if ($product->is_type('simple')) {
        $special_stock = isset($_POST['_lamkadeh_special_stock']) ? absint($_POST['_lamkadeh_special_stock']) : 0;
        $product->update_meta_data('_lamkadeh_special_stock', $special_stock);
    }
}

/**
 * Save special stock for variations
 */
add_action('woocommerce_save_product_variation', 'lamkadeh_save_variation_special_stock', 10, 2);
function lamkadeh_save_variation_special_stock($variation_id, $loop)
{
    $special_stock = isset($_POST['_lamkadeh_special_stock'][$loop]) ? absint($_POST['_lamkadeh_special_stock'][$loop]) : 0;
    update_post_meta($variation_id, '_lamkadeh_special_stock', $special_stock);
}

// ==================== FRONTEND DISPLAY ====================

/**
 * Main function to display special stock status
 */
if (!function_exists('lamkadeh_show_special_stock_status')) {
    function lamkadeh_show_special_stock_status()
    {
        global $product;

        if (!lamkadeh_should_show_special_stock($product)) {
            return;
        }

        $product_id = (int) $product->get_id();
        $is_variable = $product->is_type('variable');

        echo '<div class="lamkadeh-special-stock-wrapper"' . ($is_variable ? ' style="display: none;"' : '') . '>';

        // HTML content
        ?>
        <div class="underline-lamkadeh">
            <div class="underline-lamkadeh-line"></div>
            <div class="underline-lamkadeh-circle"></div>
        </div>
        <div class="lamkadeh-special-stock-box">
            <div class="lamkadeh-special-stock-box-title">
                <span class="lamkadeh-remaining"><?php echo esc_html__('باقیمانده: ', 'lamkadeh'); ?> <strong>0</strong></span>
                <span class="lamkadeh-sold"><?php echo esc_html__('فروخته شده: ', 'lamkadeh'); ?> <strong>0</strong></span>
            </div>
            <div class="lamkadeh-special-stock-box-progressbar">
                <div class="lamkadeh-special-stock-box-progressbar-active" style="width:0%;"></div>
            </div>
        </div>
        <?php

        // Buyers section
        $buyers_map = lamkadeh_get_buyers_for_product($product_id);
        $buyer_count = count($buyers_map);

        if ($buyer_count > 0) {
            $buyers_list = array_values($buyers_map);
            shuffle($buyers_list);
            $random_buyers = array_slice($buyers_list, 0, 3);
            $fallback_avatar = esc_url(get_template_directory_uri() . '/assets/images/theme/avatar.png');
            ?>
            <div class="lamkadeh-buyers">
                <div>
                    <?php foreach ($random_buyers as $b):
                        $avatar_url = $fallback_avatar;
                        if (isset($b['type']) && $b['type'] === 'user' && !empty($b['id'])) {
                            $custom_avatar = get_user_meta((int) $b['id'], 'lamkadeh_user_avatar', true);
                            $avatar_url = $custom_avatar ? esc_url($custom_avatar) : get_avatar_url((int) $b['id'], array('size' => 80));
                        } elseif (isset($b['type']) && $b['type'] === 'email' && !empty($b['email'])) {
                            $avatar_url = get_avatar_url($b['email'], array('size' => 80));
                        }
                        ?>
                        <img src="<?php echo esc_url($avatar_url); ?>" alt="<?php echo esc_attr__('Buyer avatar', 'lamkadeh'); ?>">
                    <?php endforeach; ?>
                </div>
                <span class="lamkadeh-buyers-title">
                    <strong><?php echo esc_html(number_format_i18n($buyer_count)); ?>+</strong>
                    <?php echo esc_html__('نفر این محصول را خریداری کرده‌اند', 'lamkadeh'); ?>
                </span>
            </div>
            <?php
        }

        echo '</div>';

        // Initialize values based on product type
        if ($is_variable) {
            $max_values = lamkadeh_get_variable_product_max_values($product);
            ?>
            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    var maxValues = <?php echo json_encode($max_values); ?>;

                    // Set initial values for variable products
                    if (maxValues.special_stock > 0) {
                        $('.lamkadeh-remaining strong').text(maxValues.remaining);
                        $('.lamkadeh-sold strong').text(maxValues.total_sold);
                        $('.lamkadeh-special-stock-box-progressbar-active').css('width', maxValues.percentage_sold + '%');
                    }
                });
            </script>
            <?php
        } else {
            // Simple product - direct values
            $special_stock = (int) get_post_meta($product_id, '_lamkadeh_special_stock', true);
            $total_sold = (int) $product->get_total_sales();
            $remaining = max(0, $special_stock - $total_sold);
            $percentage_sold = ($special_stock > 0) ? ($total_sold / $special_stock) * 100 : 0;
            $percentage_sold = min(100, max(0, round($percentage_sold, 2)));
            ?>
            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    $('.lamkadeh-remaining strong').text(<?php echo $remaining; ?>);
                    $('.lamkadeh-sold strong').text(<?php echo $total_sold; ?>);
                    $('.lamkadeh-special-stock-box-progressbar-active').css('width', '<?php echo $percentage_sold; ?>%');
                });
            </script>
            <?php
        }
    }
}

// ==================== VARIATION DATA ====================

/**
 * Add special stock data to variations
 */
add_filter('woocommerce_available_variation', 'lamkadeh_add_special_stock_to_variation_data', 10, 3);
function lamkadeh_add_special_stock_to_variation_data($data, $product, $variation)
{
    $variation_id = $variation->get_id();
    $special_stock = (int) get_post_meta($variation_id, '_lamkadeh_special_stock', true);
    // FIX: Use get_post_meta for variation sales
    $total_sold = (int) get_post_meta($variation_id, 'total_sales', true);

    // Ensure sold count doesn't exceed special stock
    $total_sold = min($total_sold, $special_stock);

    $data['lamkadeh_special_stock'] = $special_stock;
    $data['lamkadeh_total_sold'] = $total_sold;

    return $data;
}

/**
 * Add variable product data to form
 */
add_action('woocommerce_before_variations_form', 'lamkadeh_add_variable_product_data');
function lamkadeh_add_variable_product_data()
{
    global $product;

    if (!$product->is_type('variable'))
        return;

    $max_values = lamkadeh_get_variable_product_max_values($product);

    if ($max_values['special_stock'] > 0) {
        ?>
        <script type="text/javascript">
            jQuery(document).ready(function ($) {
                $('form.variations_form')
                    .data('max_special_stock', <?php echo $max_values['special_stock']; ?>)
                    .data('max_total_sold', <?php echo $max_values['total_sold']; ?>)
                    .data('max_remaining', <?php echo $max_values['remaining']; ?>)
                    .data('max_percentage_sold', <?php echo $max_values['percentage_sold']; ?>);
            });
        </script>
        <?php
    }
}

// ==================== JAVASCRIPT HANDLING ====================

/**
 * JavaScript for variable product handling - IMPROVED
 */
add_action('wp_footer', 'lamkadeh_variable_product_script');
function lamkadeh_variable_product_script()
{
    if (!is_product())
        return;
    ?>
    <script type="text/javascript">
        jQuery(document).ready(function ($) {
            function updateSpecialStockStatus(variation) {
                var $specialStockBox = $('.lamkadeh-special-stock-wrapper');

                if (!variation || !variation.lamkadeh_special_stock) {
                    // No variation selected or no special stock data
                    showMaxSpecialStock();
                    return;
                }

                var specialStock = parseInt(variation.lamkadeh_special_stock);
                var totalSold = parseInt(variation.lamkadeh_total_sold) || 0;

                // Don't show if special stock is zero or sold out
                if (specialStock <= 0 || totalSold >= specialStock) {
                    $specialStockBox.hide();
                    return;
                }

                var remaining = Math.max(0, specialStock - totalSold);
                var percentageSold = specialStock > 0 ? (totalSold / specialStock) * 100 : 0;
                percentageSold = Math.min(100, Math.max(0, Math.round(percentageSold * 100) / 100));

                // Update content
                $('.lamkadeh-remaining strong').text(remaining);
                $('.lamkadeh-sold strong').text(totalSold);
                $('.lamkadeh-special-stock-box-progressbar-active').css('width', percentageSold + '%');

                $specialStockBox.show();
            }

            function showMaxSpecialStock() {
                var $specialStockBox = $('.lamkadeh-special-stock-wrapper');
                var $form = $('form.variations_form');

                var maxSpecialStock = $form.data('max_special_stock') || 0;
                var maxTotalSold = $form.data('max_total_sold') || 0;
                var maxRemaining = $form.data('max_remaining') || 0;
                var maxPercentageSold = $form.data('max_percentage_sold') || 0;

                if (maxSpecialStock > 0 && maxRemaining > 0) {
                    $('.lamkadeh-remaining strong').text(maxRemaining);
                    $('.lamkadeh-sold strong').text(maxTotalSold);
                    $('.lamkadeh-special-stock-box-progressbar-active').css('width', maxPercentageSold + '%');
                    $specialStockBox.show();
                } else {
                    $specialStockBox.hide();
                }
            }

            // Event handlers
            $('form.variations_form')
                .on('found_variation', function (event, variation) {
                    updateSpecialStockStatus(variation);
                })
                .on('reset_data', function () {
                    showMaxSpecialStock();
                })
                .on('hide_variation', function () {
                    showMaxSpecialStock();
                });

            // Initial state
            setTimeout(function () {
                var $form = $('form.variations_form');
                if ($form.length > 0) {
                    if ($form.find('.reset_variations').is(':visible')) {
                        // Variation is selected
                        var variation = $form.data('variation');
                        updateSpecialStockStatus(variation);
                    } else {
                        // No variation selected
                        showMaxSpecialStock();
                    }
                }
            }, 100);
        });
    </script>
    <?php
}

/**
 * Update sales count when order status changes
 */
add_action('woocommerce_order_status_changed', 'lamkadeh_update_sales_on_order_status_change', 10, 4);
function lamkadeh_update_sales_on_order_status_change($order_id, $old_status, $new_status, $order)
{
    // Clear cache for all products in this order
    $items = $order->get_items();
    foreach ($items as $item) {
        $product_id = lamkadeh_get_item_product_id($item);
        if ($product_id > 0) {
            delete_transient('lamkadeh_buyers_' . $product_id);
        }
    }
}

/**
 * Clear cache when order is created/updated
 */
add_action('woocommerce_new_order', 'lamkadeh_clear_buyers_cache_on_new_order');
add_action('woocommerce_update_order', 'lamkadeh_clear_buyers_cache_on_new_order');
function lamkadeh_clear_buyers_cache_on_new_order($order_id)
{
    $order = wc_get_order($order_id);
    if (!$order)
        return;

    $items = $order->get_items();
    foreach ($items as $item) {
        $product_id = lamkadeh_get_item_product_id($item);
        if ($product_id > 0) {
            delete_transient('lamkadeh_buyers_' . $product_id);
        }
    }
}

function lamkadeh_review_display($comment, $args, $depth)
{
    $GLOBALS['comment'] = $comment;

    if (in_array($comment->comment_type, ['review', 'comment'])):
        $rating = get_comment_meta($comment->comment_ID, 'rating', true);
        $rating = $rating ? floatval($rating) : 0;

        $is_admin = user_can($comment->user_id, 'administrator');
        $is_reply = $comment->comment_parent > 0;
        $classes = 'lamkadeh-single-review-item';
        if ($is_reply) {
            $classes .= ' active';
        }
        ?>
        <div <?php comment_class($classes); ?> id="comment-<?php comment_ID(); ?>">

            <div class="lamkadeh-review-header">
                <div class="lamkadeh-review-header-right">
                    <div class="lamkadeh-review-avatar">
                        <?php echo get_avatar($comment, 64, '', '', ['class' => 'user-avatar']); ?>
                    </div>

                    <div class="lamkadeh-review-detail">
                        <div class="lamkadeh-review-detail-top">
                            <span class="lamkadeh-review-author"><?php comment_author(); ?></span>
                            <span class="lamkadeh-review-date"><?php echo get_comment_date(); ?></span>

                            <?php if (get_option('woocommerce_enable_review_rating') === 'yes' && $rating > 0): ?>
                                <div class="lamkadeh-review-rating">
                                    <span><?php echo esc_html(number_format($rating, 1)); ?></span>
                                </div>
                            <?php endif; ?>
                        </div>

                        <span class="lamkadeh-review-detail-bottom">
                            <?php
                            echo $is_admin
                                ? esc_html__("مدیر سایت", "lamkadeh")
                                : esc_html__("مشتری", "lamkadeh");
                            ?>
                        </span>
                    </div>
                </div>

                <div class="lamkadeh-review-header-left">
                    <span>“</span>
                    <span>“</span>
                </div>
            </div>

            <div class="underline-lamkadeh">
                <div class="underline-lamkadeh-line"></div>
                <div class="underline-lamkadeh-circle"></div>
            </div>

            <div class="lamkadeh-review-content">
                <?php comment_text(); ?>
            </div>

            <div class="lamkadeh-review-footer">
                <?php
                $reply_link = get_comment_reply_link(array_merge(
                    $args,
                    [
                        'reply_text' => '<span>' . esc_html__("پاسخ", "lamkadeh") . '</span>
            <svg width="20" height="20" viewBox="0 0 20 20" fill="none"
                xmlns="http://www.w3.org/2000/svg">
                <path fill-rule="evenodd" clip-rule="evenodd"
                    d="M6.27559 17.1087C6.03152 17.3528 5.63579 17.3528 5.39171 17.1087L2.89171 14.6087C2.64763 14.3646 2.64763 13.9689 2.89171 13.7248L5.39171 11.2248C5.63579 10.9807 6.03152 10.9807 6.27559 11.2248C6.51967 11.4689 6.51967 11.8646 6.27559 12.1087L4.84254 13.5417H12.5003C14.4563 13.5417 16.042 11.9561 16.042 10.0001C16.042 8.04407 14.4563 6.45842 12.5003 6.45842H6.66699C6.32181 6.45842 6.04199 6.17859 6.04199 5.83342C6.04199 5.48824 6.32181 5.20842 6.66699 5.20842H12.5003C15.1467 5.20842 17.292 7.35372 17.292 10.0001C17.292 12.6464 15.1467 14.7917 12.5003 14.7917H4.84254L6.27559 16.2248C6.51967 16.4689 6.51967 16.8646 6.27559 17.1087Z"
                    fill="#1D2977"/>
            </svg>',
                        'depth' => $depth,
                        'max_depth' => $args['max_depth'],
                        'before' => '',
                        'after' => '',
                    ]
                ));

                if ($reply_link) {
                    $reply_link = str_replace(
                        'class=\'comment-reply-link\'',
                        'class="lamkadeh-review-reply"',
                        $reply_link
                    );
                    echo $reply_link;
                }
                ?>
            </div>
        </div>
        <?php
    endif;
}

add_filter('comment_reply_link', function ($link, $args, $comment, $post) {

    $custom_class = 'lamkadeh-review-reply';

    if (strpos($link, 'class=') !== false) {
        $link = str_replace('class="', 'class="' . esc_attr($custom_class) . ' ', $link);
    } else {
        $link = str_replace('<a', '<a class="' . esc_attr($custom_class) . '"', $link);
    }

    return $link;
}, 10, 4);

add_filter('woocommerce_product_review_list_args', function ($args) {
    $args['style'] = 'div';
    $args['callback'] = 'lamkadeh_review_display';
    return $args;
});

add_filter('comment_form_fields', function ($fields) {
    $new_order = array();

    if (isset($fields['author_email_group'])) {
        $new_order['author_email_group'] = $fields['author_email_group'];
    }

    foreach ($fields as $key => $value) {
        if (!isset($new_order[$key])) {
            $new_order[$key] = $value;
        }
    }

    return $new_order;
});

add_action('woocommerce_after_single_product_summary', 'lamkadeh_insert_custom_banner_product', 4);
function lamkadeh_insert_custom_banner_product()
{
    global $product;

    $banner_content = get_post_meta($product->get_id(), '_short_code_banner_product', true);

    if (!empty($banner_content)) {
        echo do_shortcode($banner_content);
    }
}

function format_price($price)
{
    return number_format(floatval($price), 0, '.', ',');
}

add_filter('preprocess_comment', 'lamkadeh_merge_name_lastname_in_comment');
function lamkadeh_merge_name_lastname_in_comment($commentdata)
{
    if (isset($_POST['author']) && isset($_POST['last_name'])) {
        $first_name = sanitize_text_field($_POST['author']);
        $last_name = sanitize_text_field($_POST['last_name']);

        $commentdata['comment_author'] = trim($first_name . ' ' . $last_name);
    }
    return $commentdata;
}

add_filter('woocommerce_output_related_products_args', function ($args) {
    $args['posts_per_page'] = 8;
    $args['columns'] = 4;
    return $args;
});
