<?php
class Lamkadeh_Filter_Product_Brand_Widget extends WP_Widget
{

    public function __construct()
    {
        parent::__construct(
            'lamkadeh_filter_product_brand',
            __('فیلتر برند محصولات - لمکده', 'lamkadeh'),
            array('description' => __('نمایش دکمه فیلتر برند در سایدبار فروشگاه', 'lamkadeh'))
        );

        // اضافه کردن Ajax actions
        add_action('wp_ajax_filter_brands_search', array($this, 'ajax_search_brands'));
        add_action('wp_ajax_nopriv_filter_brands_search', array($this, 'ajax_search_brands'));

        // اضافه کردن فیلتر برای کوئری محصولات
        add_action('woocommerce_product_query', array($this, 'filter_products_by_brand'));
    }

    // تابع فیلتر کردن محصولات بر اساس برند
    public function filter_products_by_brand($q)
    {
        $selected_brands = $this->get_selected_brands_from_url();

        if (!empty($selected_brands)) {
            $tax_query = $q->get('tax_query');

            if (!is_array($tax_query)) {
                $tax_query = array();
            }

            // اضافه کردن فیلتر برند به tax_query
            $brand_tax_query = array(
                array(
                    'taxonomy' => 'product_brand', // یا 'pa_brand'
                    'field'    => 'slug',
                    'terms'    => $selected_brands,
                    'operator' => 'IN'
                )
            );

            // اگر tax_query وجود داره، با رابطه AND اضافه کن
            if (!empty($tax_query)) {
                $tax_query = array(
                    'relation' => 'AND',
                    $tax_query,
                    $brand_tax_query
                );
            } else {
                $tax_query = $brand_tax_query;
            }

            $q->set('tax_query', $tax_query);
        }
    }

    // تابع دریافت برندهای انتخاب شده از URL
    private function get_selected_brands_from_url()
    {
        $selected_brands = array();

        if (isset($_GET['brand'])) {
            if (is_array($_GET['brand'])) {
                $selected_brands = array_map('sanitize_text_field', $_GET['brand']);
            } else {
                $selected_brands = array_map('trim', explode(',', sanitize_text_field($_GET['brand'])));
            }
        }

        return array_filter($selected_brands);
    }

    // تابع دریافت ID دسته‌بندی فعلی
    private function get_current_category_id()
    {
        if (is_product_category()) {
            $category = get_queried_object();
            return $category->term_id;
        }
        return 0;
    }

    // تابع جستجوی Ajax
    public function ajax_search_brands()
    {
        $search_term = sanitize_text_field($_POST['search_term'] ?? '');
        $selected_brands = isset($_POST['selected_brands']) ? (array)$_POST['selected_brands'] : array();
        $category_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : 0;

        // دریافت تنظیمات از ویجت
        $widget_options = get_option('widget_lamkadeh_filter_product_brand');
        $hide_empty = true; // مقدار پیش‌فرض

        if (!empty($widget_options) && isset($widget_options[2]['hide_empty'])) {
            $hide_empty = (bool)$widget_options[2]['hide_empty'];
        }

        $brands = $this->get_product_brands($search_term, $category_id, $hide_empty);

        ob_start();
        $this->render_brands_list($brands, $selected_brands);
        $html = ob_get_clean();

        wp_send_json_success(array(
            'html' => $html,
            'count' => count($brands)
        ));
    }

    // تابع دریافت برندها از ووکامرس با فیلتر دسته‌بندی
    private function get_product_brands($search_term = '', $category_id = 0, $hide_empty = true)
    {
        $taxonomy = 'product_brand'; // یا 'pa_brand' بسته به تنظیمات شما

        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => $hide_empty, // استفاده از تنظیمات کاربر
        );

        // اگر دسته‌بندی مشخص شده، کوئری محصولات رو فیلتر کن
        if ($category_id > 0) {
            // اول همه برندها رو بگیر
            $all_brands = get_terms($args);

            if (is_wp_error($all_brands) || empty($all_brands)) {
                return array();
            }

            // حالا برای هر برند بررسی کن که در این دسته‌بندی محصول داره یا نه
            $brands = array();
            foreach ($all_brands as $term) {
                // کوئری برای محصولاتی که هم در این برند و هم در این دسته‌بندی هستن
                $products_count = $this->get_products_count_by_brand_and_category($term->term_id, $category_id);

                // اگر hide_empty فعال هست و محصولی نداره، ردش کن
                if ($hide_empty && $products_count === 0) {
                    continue;
                }

                // اگر جستجو انجام شده، بررسی کن نام برند شامل عبارت جستجو هست یا نه
                if (!empty($search_term) && stripos($term->name, $search_term) === false) {
                    continue;
                }

                $brands[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                    'count' => $products_count,
                    'url' => get_term_link($term)
                );
            }
        } else {
            // اگر دسته‌بندی مشخص نشده، همه برندها رو بگیر
            if (!empty($search_term)) {
                $args['name__like'] = $search_term;
            }

            $terms = get_terms($args);

            if (is_wp_error($terms)) {
                return array();
            }

            $brands = array();
            foreach ($terms as $term) {
                $brands[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                    'count' => $term->count,
                    'url' => get_term_link($term)
                );
            }
        }

        // مرتب‌سازی بر اساس نام
        usort($brands, function ($a, $b) {
            return strcmp($a['name'], $b['name']);
        });

        return $brands;
    }

    // تابع شمارش محصولات در یک برند و دسته‌بندی خاص
    private function get_products_count_by_brand_and_category($brand_term_id, $category_id)
    {
        $taxonomy = 'product_brand'; // یا 'pa_brand' بسته به تنظیمات شما

        $args = array(
            'status' => 'publish',
            'limit' => -1,
            'return' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $brand_term_id,
                ),
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $category_id,
                )
            )
        );

        $products = wc_get_products($args);
        return count($products);
    }

    // تابع رندر لیست برندها
    private function render_brands_list($brands, $selected_brands = array())
    {
        if (empty($brands)) {
            echo '<div class="no-brands-message">' . __('برندی در این دسته‌بندی یافت نشد.', 'lamkadeh') . '</div>';
            return;
        }

        foreach ($brands as $brand) {
            $is_active = in_array($brand['slug'], $selected_brands);
?>
            <div class="content-brand-item <?php echo $is_active ? 'active' : ''; ?>"
                data-brand-slug="<?php echo esc_attr($brand['slug']); ?>"
                data-brand-name="<?php echo esc_attr($brand['name']); ?>">
                <div class="content-brand-item-data">
                    <span><?php echo esc_html($brand['name']); ?></span>
                    <span><?php echo $brand['count'] . ' ' . __('کالا', 'lamkadeh'); ?></span>
                </div>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <circle cx="12" cy="12" r="10" stroke="white" stroke-width="1.5" />
                    <path d="M8.5 12.5L10.5 14.5L15.5 9.5" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" />
                </svg>
            </div>
        <?php
        }
    }

    public function widget($args, $instance)
    {
        // دریافت تنظیمات
        $hide_empty = !empty($instance['hide_empty']) ? (bool)$instance['hide_empty'] : true;

        // دریافت برندهای انتخاب شده از URL
        $selected_brands = $this->get_selected_brands_from_url();
        $current_category_id = $this->get_current_category_id();
        $all_brands = $this->get_product_brands('', $current_category_id, $hide_empty);

        // ایجاد لیست برندها برای JavaScript
        $brands_data = array();
        foreach ($all_brands as $brand) {
            $brands_data[$brand['slug']] = $brand['name'];
        }

        // بررسی آیا برندی انتخاب شده یا نه
        $has_selected_brands = !empty($selected_brands);

        echo $args['before_widget'];
        ?>

        <section class="filter-product-brand">
            <div class="filter-product-brand-buttom <?php echo $has_selected_brands ? 'active' : ''; ?>">
                <span><?php echo __("برند", "lamkadeh"); ?></span>
                <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M21.0813 15.96C20.8597 15.96 20.638 16.0417 20.463 16.2167L13.9997 22.68L7.53632 16.2167C7.19798 15.8783 6.63798 15.8783 6.29965 16.2167C5.96132 16.555 5.96132 17.115 6.29965 17.4533L13.3813 24.535C13.7197 24.8733 14.2796 24.8733 14.618 24.535L21.6997 17.4533C22.038 17.115 22.038 16.555 21.6997 16.2167C21.5363 16.0417 21.303 15.96 21.0813 15.96Z" fill="white" />
                    <path d="M14 3.20833C13.5217 3.20833 13.125 3.60499 13.125 4.08333L13.125 23.7183C13.125 24.1967 13.5217 24.5933 14 24.5933C14.4783 24.5933 14.875 24.1967 14.875 23.7183L14.875 4.08333C14.875 3.60499 14.4783 3.20833 14 3.20833Z" fill="white" />
                </svg>

                <div class="content">
                    <!-- بخش برندهای انتخاب شده -->
                    <div class="content-brand-me" id="selected-brands-container" <?php echo empty($selected_brands) ? 'style="display:none;"' : ''; ?>>
                        <span><?php echo __("انتخــاب شمـــــــــا", "lamkadeh") ?></span>
                        <div class="content-brand-me-items" id="selected-brands-items">
                            <?php
                            foreach ($selected_brands as $brand_slug):
                                // فقط برندهایی رو نمایش بده که در دسته‌بندی فعلی موجود هستن
                                if (isset($brands_data[$brand_slug])):
                                    $brand_term = get_term_by('slug', $brand_slug, 'product_brand');
                                    if ($brand_term):
                            ?>
                                        <div class="content-brand-me-item" data-brand-slug="<?php echo esc_attr($brand_slug); ?>">
                                            <span><?php echo esc_html($brand_term->name); ?></span>
                                            <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M8.35891 7.4747C8.11483 7.23062 7.7191 7.23062 7.47502 7.4747C7.23095 7.71878 7.23095 8.11451 7.47502 8.35858L9.11642 9.99998L7.47504 11.6414C7.23096 11.8854 7.23096 12.2812 7.47504 12.5253C7.71912 12.7693 8.11485 12.7693 8.35893 12.5253L10.0003 10.8839L11.6417 12.5252C11.8858 12.7693 12.2815 12.7693 12.5256 12.5252C12.7696 12.2812 12.7696 11.8854 12.5256 11.6414L10.8842 9.99998L12.5256 8.3586C12.7697 8.11452 12.7697 7.7188 12.5256 7.47472C12.2815 7.23064 11.8858 7.23064 11.6417 7.47472L10.0003 9.1161L8.35891 7.4747Z" fill="#1D2977" />
                                                <path fill-rule="evenodd" clip-rule="evenodd" d="M10.0003 1.04166C5.05277 1.04166 1.04199 5.05244 1.04199 9.99999C1.04199 14.9475 5.05277 18.9583 10.0003 18.9583C14.9479 18.9583 18.9587 14.9475 18.9587 9.99999C18.9587 5.05244 14.9479 1.04166 10.0003 1.04166ZM2.29199 9.99999C2.29199 5.74279 5.74313 2.29166 10.0003 2.29166C14.2575 2.29166 17.7087 5.74279 17.7087 9.99999C17.7087 14.2572 14.2575 17.7083 10.0003 17.7083C5.74313 17.7083 2.29199 14.2572 2.29199 9.99999Z" fill="#1D2977" />
                                            </svg>
                                        </div>
                            <?php
                                    endif;
                                endif;
                            endforeach;
                            ?>
                        </div>
                    </div>

                    <span><?php echo __("برنــدهـــــــــا", "lamkadeh") ?></span>

                    <!-- بخش جستجو -->
                    <div class="filter-product-brand-search">
                        <input type="text" placeholder="<?php echo __("جستوجو در برند ها", "lamkadeh") ?>">
                        <svg width="24" height="24" viewBox="0 0 20 30" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path fill-rule="evenodd" clip-rule="evenodd" d="M11 2.75C6.44365 2.75 2.75 6.44365 2.75 11C2.75 15.5563 6.44365 19.25 11 19.25C15.5563 19.25 19.25 15.5563 19.25 11C19.25 6.44365 15.5563 2.75 11 2.75ZM1.25 11C1.25 5.61522 5.61522 1.25 11 1.25C16.3848 1.25 20.75 5.61522 20.75 11C20.75 16.3848 16.3848 20.75 11 20.75C5.61522 20.75 1.25 16.3848 1.25 11ZM20.1579 19.7511C19.9264 19.7335 19.7335 19.9264 19.7511 20.1579C19.7514 20.1592 19.7553 20.1848 19.7746 20.2573C19.7974 20.3424 19.8312 20.4554 19.8828 20.6277C19.9301 20.7857 19.9609 20.8881 19.9862 20.9641C20.0121 21.0419 20.021 21.0568 20.0171 21.0496C20.1225 21.2465 20.3745 21.31 20.5607 21.1867C20.5538 21.1912 20.5688 21.1824 20.6284 21.1261C20.6868 21.0712 20.7624 20.9957 20.8791 20.8791C20.9957 20.7624 21.0712 20.6868 21.1261 20.6284C21.1727 20.579 21.1868 20.5602 21.1877 20.5592C21.3093 20.3736 21.2463 20.1236 21.0511 20.018C21.0499 20.0175 21.0287 20.0077 20.9641 19.9862C20.8881 19.9609 20.7857 19.9301 20.6277 19.8828C20.4554 19.8312 20.3424 19.7974 20.2573 19.7746C20.1848 19.7553 20.1591 19.7514 20.1579 19.7511ZM18.2564 20.2833C18.1612 19.1267 19.1267 18.1612 20.2833 18.2564C20.4832 18.2728 20.7251 18.3456 20.9862 18.4242C21.0101 18.4314 21.0341 18.4387 21.0583 18.4459C21.0801 18.4524 21.1018 18.4589 21.1234 18.4654C21.3632 18.5369 21.5881 18.604 21.7576 18.6948C22.7335 19.2173 23.0485 20.4659 22.4373 21.3889C22.3312 21.5492 22.165 21.715 21.9878 21.8917C21.9719 21.9076 21.9558 21.9236 21.9397 21.9397C21.9236 21.9558 21.9076 21.9719 21.8917 21.9878C21.7149 22.165 21.5492 22.3312 21.3889 22.4373C20.4659 23.0485 19.2173 22.7335 18.6948 21.7576C18.604 21.5881 18.5369 21.3632 18.4654 21.1234C18.4589 21.1018 18.4524 21.0801 18.4459 21.0583C18.4387 21.0341 18.4314 21.0101 18.4242 20.9862C18.3457 20.7252 18.2728 20.4833 18.2564 20.2833Z" fill="#020308" fill-opacity="0.64" />
                        </svg>
                    </div>

                    <div class="underline-lamkadeh">
                        <div class="underline-lamkadeh-line"></div>
                        <div class="underline-lamkadeh-circle"></div>
                    </div>

                    <!-- لیست برندها -->
                    <div class="content-brand">
                        <?php $this->render_brands_list($all_brands, $selected_brands); ?>
                    </div>
                </div>
            </div>
        </section>

        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const filterWidget = {
                    init() {
                        // مقداردهی اولیه از URL
                        this.selectedBrands = this.getSelectedBrandsFromURL();
                        this.brandsData = <?php echo json_encode($brands_data); ?>;
                        this.currentCategoryId = <?php echo $current_category_id; ?>;

                        // اگر برندی انتخاب شده، آکوردین رو active کن
                        if (this.selectedBrands.length > 0) {
                            this.openAccordion();
                        }

                        this.bindEvents();
                        this.updateSelectedBrandsUI();
                    },

                    // تابع باز کردن آکوردین
                    openAccordion() {
                        const accordionButton = document.querySelector('.filter-product-brand-buttom');
                        accordionButton.classList.add('active');
                    },

                    // تابع بستن آکوردین
                    closeAccordion() {
                        const accordionButton = document.querySelector('.filter-product-brand-buttom');
                        accordionButton.classList.remove('active');
                    },

                    // تابع دریافت برندهای انتخاب شده از URL
                    getSelectedBrandsFromURL() {
                        const urlParams = new URLSearchParams(window.location.search);
                        const brandParam = urlParams.get('brand');
                        if (brandParam) {
                            return brandParam.split(',').filter(slug => slug.trim() !== '');
                        }
                        return [];
                    },

                    bindEvents() {
                        // آکاردیون
                        const accordionButton = document.querySelector('.filter-product-brand-buttom');
                        const content = accordionButton.querySelector('.content');

                        accordionButton.addEventListener('click', function(e) {
                            if (content.contains(e.target)) {
                                return;
                            }
                            if (e.target.tagName.toLowerCase() === 'svg' ||
                                e.target.tagName.toLowerCase() === 'path' ||
                                e.target.tagName.toLowerCase() === 'span') {
                                this.classList.toggle('active');
                            }
                        });

                        // جستجوی Ajax
                        const searchInput = document.querySelector('.filter-product-brand-search input');
                        let searchTimeout;

                        searchInput.addEventListener('input', (e) => {
                            clearTimeout(searchTimeout);
                            searchTimeout = setTimeout(() => {
                                this.searchBrands(e.target.value);
                            }, 300);
                        });

                        // کلیک روی برند
                        document.addEventListener('click', (e) => {
                            const brandItem = e.target.closest('.content-brand-item');
                            if (brandItem) {
                                this.toggleBrand(brandItem);
                            }

                            // حذف برند
                            const removeBtn = e.target.closest('.content-brand-me-item');
                            if (removeBtn) {
                                this.removeBrand(removeBtn.dataset.brandSlug);
                            }
                        });
                    },

                    searchBrands(searchTerm) {
                        const data = new FormData();
                        data.append('action', 'filter_brands_search');
                        data.append('search_term', searchTerm);
                        data.append('selected_brands', this.selectedBrands);
                        data.append('category_id', this.currentCategoryId);

                        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                method: 'POST',
                                body: data
                            })
                            .then(response => response.json())
                            .then(result => {
                                if (result.success) {
                                    document.querySelector('.content-brand').innerHTML = result.data.html;
                                }
                            })
                            .catch(error => console.error('Error:', error));
                    },

                    toggleBrand(brandItem) {
                        const brandSlug = brandItem.dataset.brandSlug;

                        if (this.selectedBrands.includes(brandSlug)) {
                            // حذف برند
                            this.selectedBrands = this.selectedBrands.filter(slug => slug !== brandSlug);
                            brandItem.classList.remove('active');
                        } else {
                            // اضافه کردن برند
                            this.selectedBrands.push(brandSlug);
                            brandItem.classList.add('active');
                        }

                        this.updateSelectedBrandsUI();

                        // اگر برندی انتخاب شده، آکوردین رو باز کن
                        if (this.selectedBrands.length > 0) {
                            this.openAccordion();
                        }

                        this.updateURL();
                    },

                    removeBrand(brandSlug) {
                        this.selectedBrands = this.selectedBrands.filter(slug => slug !== brandSlug);

                        // به‌روزرسانی وضعیت آیتم در لیست
                        const brandItem = document.querySelector(`.content-brand-item[data-brand-slug="${brandSlug}"]`);
                        if (brandItem) {
                            brandItem.classList.remove('active');
                        }

                        this.updateSelectedBrandsUI();

                        // اگر هیچ برندی انتخاب نشده، آکوردین رو ببند
                        if (this.selectedBrands.length === 0) {
                            this.closeAccordion();
                        }

                        this.updateURL();
                    },

                    updateSelectedBrandsUI() {
                        const container = document.getElementById('selected-brands-container');
                        const itemsContainer = document.getElementById('selected-brands-items');

                        itemsContainer.innerHTML = '';

                        // فقط برندهایی رو نمایش بده که در brandsData موجود هستن
                        const validSelectedBrands = this.selectedBrands.filter(slug => this.brandsData[slug]);

                        if (validSelectedBrands.length === 0) {
                            container.style.display = 'none';
                            return;
                        }

                        container.style.display = 'block';

                        validSelectedBrands.forEach(slug => {
                            const brandName = this.brandsData[slug];

                            if (brandName) {
                                const item = document.createElement('div');
                                item.className = 'content-brand-me-item';
                                item.dataset.brandSlug = slug;
                                item.innerHTML = `
                                <span>${brandName}</span>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M8.35891 7.4747C8.11483 7.23062 7.7191 7.23062 7.47502 7.4747C7.23095 7.71878 7.23095 8.11451 7.47502 8.35858L9.11642 9.99998L7.47504 11.6414C7.23096 11.8854 7.23096 12.2812 7.47504 12.5253C7.71912 12.7693 8.11485 12.7693 8.35893 12.5253L10.0003 10.8839L11.6417 12.5252C11.8858 12.7693 12.2815 12.7693 12.5256 12.5252C12.7696 12.2812 12.7696 11.8854 12.5256 11.6414L10.8842 9.99998L12.5256 8.3586C12.7697 8.11452 12.7697 7.7188 12.5256 7.47472C12.2815 7.23064 11.8858 7.23064 11.6417 7.47472L10.0003 9.1161L8.35891 7.4747Z" fill="#1D2977" />
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M10.0003 1.04166C5.05277 1.04166 1.04199 5.05244 1.04199 9.99999C1.04199 14.9475 5.05277 18.9583 10.0003 18.9583C14.9479 18.9583 18.9587 14.9475 18.9587 9.99999C18.9587 5.05244 14.9479 1.04166 10.0003 1.04166ZM2.29199 9.99999C2.29199 5.74279 5.74313 2.29166 10.0003 2.29166C14.2575 2.29166 17.7087 5.74279 17.7087 9.99999C17.7087 14.2572 14.2575 17.7083 10.0003 17.7083C5.74313 17.7083 2.29199 14.2572 2.29199 9.99999Z" fill="#1D2977" />
                                </svg>
                            `;
                                itemsContainer.appendChild(item);
                            }
                        });
                    },

                    updateURL() {
                        const url = new URL(window.location.href);

                        // فقط برندهای معتبر رو نگه دار
                        const validSelectedBrands = this.selectedBrands.filter(slug => this.brandsData[slug]);

                        if (validSelectedBrands.length > 0) {
                            url.searchParams.set('brand', validSelectedBrands.join(','));
                        } else {
                            url.searchParams.delete('brand');
                        }

                        // رفرش صفحه با URL جدید
                        window.location.href = url.toString();
                    }
                };

                filterWidget.init();
            });
        </script>

    <?php
        echo $args['after_widget'];
    }

    public function form($instance)
    {
        // مقداردهی پیش‌فرض
        $hide_empty = isset($instance['hide_empty']) ? (bool)$instance['hide_empty'] : true;
    ?>
        <p>
            <input type="checkbox"
                id="<?php echo $this->get_field_id('hide_empty'); ?>"
                name="<?php echo $this->get_field_name('hide_empty'); ?>"
                value="1"
                <?php checked($hide_empty, true); ?> />
            <label for="<?php echo $this->get_field_id('hide_empty'); ?>">
                <?php echo __('مخفی کردن برندهای خالی', 'lamkadeh'); ?>
            </label>
            <br>
            <small><?php echo __('اگر فعال باشد، برندهایی که محصولی ندارند نمایش داده نمی‌شوند.', 'lamkadeh'); ?></small>
        </p>

        <p><?php echo __('این ویجت فیلتر برند محصولات را نمایش می‌دهد.', 'lamkadeh'); ?></p>
<?php
    }

    public function update($new_instance, $old_instance)
    {
        $instance = array();
        $instance['hide_empty'] = !empty($new_instance['hide_empty']) ? 1 : 0;
        return $instance;
    }
}

add_action('widgets_init', function () {
    register_widget('Lamkadeh_Filter_Product_Brand_Widget');
});

// اضافه کردن فیلتر برای صفحات آرشیو محصولات
add_filter('woocommerce_shortcode_products_query', 'lamkadeh_filter_shortcode_products_by_brand', 10, 2);
function lamkadeh_filter_shortcode_products_by_brand($query_args, $attributes)
{
    $selected_brands = lamkadeh_get_selected_brands_from_url();

    if (!empty($selected_brands)) {
        if (!isset($query_args['tax_query'])) {
            $query_args['tax_query'] = array();
        }

        $query_args['tax_query'][] = array(
            'taxonomy' => 'product_brand', // یا 'pa_brand'
            'field'    => 'slug',
            'terms'    => $selected_brands,
            'operator' => 'IN'
        );
    }

    return $query_args;
}

// تابع کمکی برای دریافت برندهای انتخاب شده
function lamkadeh_get_selected_brands_from_url()
{
    $selected_brands = array();

    if (isset($_GET['brand'])) {
        if (is_array($_GET['brand'])) {
            $selected_brands = array_map('sanitize_text_field', $_GET['brand']);
        } else {
            $selected_brands = array_map('trim', explode(',', sanitize_text_field($_GET['brand'])));
        }
    }

    return array_filter($selected_brands);
}
